/**
 * @license
 * Copyright 2018 Google LLC. All Rights Reserved.
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * =============================================================================
 */
import { env } from '@tensorflow/tfjs-core';
const contexts = {};
const WEBGL_ATTRIBUTES = {
    alpha: false,
    antialias: false,
    premultipliedAlpha: false,
    preserveDrawingBuffer: false,
    depth: false,
    stencil: false,
    failIfMajorPerformanceCaveat: true
};
export function clearWebGLContext(webGLVersion) {
    delete contexts[webGLVersion];
}
export function setWebGLContext(webGLVersion, gl) {
    contexts[webGLVersion] = gl;
}
export function getWebGLContext(webGLVersion, customCanvas) {
    if (!(webGLVersion in contexts) || customCanvas != null) {
        const newCtx = getWebGLRenderingContext(webGLVersion, customCanvas);
        if (newCtx !== null) {
            contexts[webGLVersion] = newCtx;
        }
        else {
            console.log('Could not get context for WebGL version', webGLVersion);
            return null;
        }
    }
    const gl = contexts[webGLVersion];
    if (gl == null || gl.isContextLost()) {
        delete contexts[webGLVersion];
        return getWebGLContext(webGLVersion);
    }
    gl.disable(gl.DEPTH_TEST);
    gl.disable(gl.STENCIL_TEST);
    gl.disable(gl.BLEND);
    gl.disable(gl.DITHER);
    gl.disable(gl.POLYGON_OFFSET_FILL);
    gl.disable(gl.SAMPLE_COVERAGE);
    gl.enable(gl.SCISSOR_TEST);
    gl.enable(gl.CULL_FACE);
    gl.cullFace(gl.BACK);
    return contexts[webGLVersion];
}
function createCanvas(webGLVersion) {
    // Use canvas element for Safari, since its offscreen canvas does not support
    // fencing.
    if (!env().getBool('IS_SAFARI') && typeof OffscreenCanvas !== 'undefined' &&
        webGLVersion === 2) {
        return new OffscreenCanvas(300, 150);
    }
    else if (typeof document !== 'undefined') {
        return document.createElement('canvas');
    }
    else {
        throw new Error('Cannot create a canvas in this context');
    }
}
function getWebGLRenderingContext(webGLVersion, customCanvas) {
    if (webGLVersion !== 1 && webGLVersion !== 2) {
        throw new Error('Cannot get WebGL rendering context, WebGL is disabled.');
    }
    const canvas = customCanvas == null ? createCanvas(webGLVersion) : customCanvas;
    canvas.addEventListener('webglcontextlost', (ev) => {
        ev.preventDefault();
        delete contexts[webGLVersion];
    }, false);
    if (env().getBool('SOFTWARE_WEBGL_ENABLED')) {
        WEBGL_ATTRIBUTES.failIfMajorPerformanceCaveat = false;
    }
    if (webGLVersion === 1) {
        return (
        // tslint:disable-next-line
        canvas.getContext('webgl', WEBGL_ATTRIBUTES) ||
            canvas
                .getContext('experimental-webgl', WEBGL_ATTRIBUTES));
    }
    return canvas.getContext('webgl2', WEBGL_ATTRIBUTES);
}
//# sourceMappingURL=data:application/json;base64,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